package oracle.spatial.georaster.samples;

import com.sun.media.jai.codec.ImageCodec;
import com.sun.media.jai.codec.ImageEncoder;
import com.sun.media.jai.codec.TIFFEncodeParam;

import java.awt.BorderLayout;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.RenderedImage;

import java.io.File;
import java.io.FileOutputStream;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import oracle.spatial.georaster.JGeoRaster;


/**
 * This is a sample java class, which shows how to implement a simple exporter to export a GeoRaster object
 * as a tiff image based on GeoRaster Java API library.
 * <p>
 */
public class Exporter
{
    private JGeoRaster   m_jgeor=null;
    
    /**
     * constructor
     */
    public Exporter(JGeoRaster jgeor)
    {
        m_jgeor=jgeor;
    }
    
    public void showDialog()
    {
        GeoExporterDialog dialog=new GeoExporterDialog();
        dialog.show();
    }
    /**
     * This inner class is a customized Java JDialog to get input and generate a tiff image.
     * <p>
     */    
    class GeoExporterDialog extends JDialog implements ActionListener
    {
        JComboBox m_fileType = new JComboBox();
        JTextField m_fileName=new JTextField();
        JButton m_browse=new JButton();

        String m_currentDir=null;
        boolean m_ret=false;

        public GeoExporterDialog() throws HeadlessException
        {
            super();
            init();
        }

        private void init()
        {
            setModal(true);
            setTitle("Export to a file");
            setSize(500, 140);
            setResizable(false);

            JPanel tmpPanel = new JPanel(new BorderLayout());
            JPanel mainPanel = new JPanel(new GridBagLayout());
            mainPanel.setBorder(new javax.swing.border.EmptyBorder(6, 6, 0, 6));
            tmpPanel.add(mainPanel, BorderLayout.NORTH);
            JLabel fileTypeL = new JLabel("File type:");
            mainPanel.add(fileTypeL, new GridBagConstraints(
                0, // gridx
                0, // gridy
                2, // gridwidth
                1, // gridheight
                0.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(3, 6, 3, 3), // insets
                0, // ipadx
                0 // ipady
                ));
            m_fileType.addItem("TIFF");
            mainPanel.add(m_fileType, new GridBagConstraints(
                2, // gridx
                0, // gridy
                6, // gridwidth
                1, // gridheight
                1.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(3, 3, 3, 6), // insets
                0, // ipadx
                0 // ipady
                ));
            fileTypeL.setDisplayedMnemonic('f');
            fileTypeL.setLabelFor(m_fileType);
    
            JLabel fileNameL = new JLabel("File name:");
            mainPanel.add(fileNameL, new GridBagConstraints(
                0, // gridx
                1, // gridy
                2, // gridwidth
                1, // gridheight
                0.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(3, 6, 3, 3), // insets
                0, // ipadx
                0 // ipady
                ));
            m_fileName.setText("");
            mainPanel.add(m_fileName, new GridBagConstraints(
                2, // gridx
                1, // gridy
                5, // gridwidth
                1, // gridheight
                1.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(3, 3, 3, 6), // insets
                0, // ipadx
                0 // ipady
                ));
            fileNameL.setDisplayedMnemonic('i');
            fileNameL.setLabelFor(m_fileName);
            m_browse.setMnemonic('B');
            m_browse.setText("Browse");
            mainPanel.add(m_browse, new GridBagConstraints(
                7, // gridx
                1, // gridy
                1, // gridwidth
                1, // gridheight
                0.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(3, 3, 3, 6), // insets
                0, // ipadx
                0 // ipady
                ));
    
            mainPanel.setBounds(6, 6, 6, 6);
            mainPanel.setBorder(BorderFactory.createRaisedBevelBorder());
    
            JPanel butPanel = new JPanel(new GridBagLayout());
            JButton okButton = new JButton("  OK   ");
            JButton cancelButton = new JButton("Cancel");
    
            okButton.addActionListener(new java.awt.event.ActionListener() {
              public void actionPerformed(ActionEvent e)
              {
                okButton_actionPerformed(e);
              }
            });
    
            cancelButton.addActionListener(new java.awt.event.ActionListener() {
              public void actionPerformed(ActionEvent e)
              {
                dispose();
              }
            });
    
            butPanel.add(okButton, new GridBagConstraints(
                0, // gridx
                0, // gridy
                1, // gridwidth
                1, // gridheight
                0.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(0, 0, 6, 3), // insets - 0,0,8,4
                0, // ipadx
                0 // ipady
                ));
            butPanel.add(cancelButton, new GridBagConstraints(
                1, // gridx
                0, // gridy
                1, // gridwidth
                1, // gridheight
                0.0D, // weightx
                0.0D, // weighty
                GridBagConstraints.CENTER, // anchor
                GridBagConstraints.HORIZONTAL, // fill
                new Insets(0, 3, 6, 6), // insets - 0,0,8,4
                0, // ipadx
                0 // ipady
                ));
    
            okButton.requestFocus();
            JPanel panel = new JPanel(new BorderLayout());
            panel.add(butPanel, BorderLayout.EAST);
            panel.setBorder(BorderFactory.createRaisedBevelBorder());
            tmpPanel.add(panel, BorderLayout.SOUTH);
            getContentPane().add(tmpPanel);
    
            m_browse.addActionListener(this);
    
            okButton.setDefaultCapable(true);
            rootPane.setDefaultButton(okButton);
    
            this.setLocation(150,150);
        }

        public void actionPerformed(ActionEvent e)
        {
            if (e.getSource() == this.m_browse)
            {
                JFileChooser chooser=new JFileChooser(m_currentDir);
                ImageFileFilter filter = new ImageFileFilter(0);
                chooser.setFileFilter(filter);
                int returnVal = chooser.showOpenDialog(this);
                if (returnVal == JFileChooser.APPROVE_OPTION)
                {
                    m_fileName.setText(chooser.getSelectedFile().getAbsolutePath());
                    m_currentDir=chooser.getCurrentDirectory().getAbsolutePath();
                }
            }
        }

        private void okButton_actionPerformed(ActionEvent e)
        {
            // check the validity of the entered values.
            String msg = checkForNullValues();
            if (msg != null)
            {
                JOptionPane.showConfirmDialog(this,
                                        msg, "Error!",
                                        javax.swing.JOptionPane.DEFAULT_OPTION);
            return;
            }

            try
            {
                RenderedImage image=m_jgeor.getGeoRasterImageObject().getRasterImage(0);
                String fileName=m_fileName.getText().trim();
                if(!fileName.toLowerCase().endsWith(".tif"))
                    fileName=fileName+".tif";
                FileOutputStream out = new FileOutputStream(fileName);
                TIFFEncodeParam encodeParam = new TIFFEncodeParam();
                encodeParam.setCompression(TIFFEncodeParam.COMPRESSION_NONE);
                ImageEncoder encoder = ImageCodec.createImageEncoder("TIFF", out, encodeParam);
                encoder.encode(image);
                out.close();
            }
            catch (Exception ee)
            {
                System.out.println("IOException at encoding...");
            }            
            dispose();
        }

        // check any not-null constraint violations.
        // returns a string of comma delimited fields that are null
        private String checkForNullValues()
        {
            String data = m_fileName.getText().trim().toLowerCase();
            if (data == null || data.trim().length() == 0)
            {
              return ("Missing values for : File name");
            }
            return null;
        }
    }
    /**
     * This inner class is a TIFF image file filter.
     * <p>
     * @author  zzhang
     * @version 11
     */    
    class ImageFileFilter extends javax.swing.filechooser.FileFilter
    {
        private int m_type = 0;
        public ImageFileFilter(int type)
        {
            m_type = type;
        }

        public boolean accept(File f)
        {
            if (m_type == 0)
            {
                //for images
                if (f.getName().toLowerCase().endsWith("tif"))
                {
                    return true;
                }
                if (f.isDirectory())
                {
                    return true;
                }
            }
            if (m_type == 1)
            {
                //for world file
                return true;
            }
            return false;
        }

        public String getDescription()
        {
            if (m_type == 0)
            {
                //for images
                return "image files";
            }
            if(m_type == 1)
            {
                //for world file
                return "GeoRef files";
            }
            return "";
        }
    }
}
